--[[---------------------------------------------------------------------------
	Chocolatier Two Shop UI
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

local priceFont = { standardFontName, kEighteen, DefaultColor }
local countFont = { standardFontName, kSixteen, DefaultColor }

local self = gDialogTable.shop
local char = gDialogTable.char or self.character[1]

gShopItems = gDialogTable.items
local itemCount = table.getn(gShopItems)

local transactionMade = false

local items = {}

-------------------------------------------------------------------------------

local insetX = 30
local insetY = 60
local windowX = ui.xOffset + insetX
local windowY = ui.yOffset + insetY
local windowW = ui.screenWidth - 2 * insetX
local windowH = 240		-- 330	--350

if itemCount > 14 then
	windowH = windowH + 25
	priceFont[2] = kTwentyThree
end

if itemCount > 20 then
	priceFont[2] = kEighteen
end

if itemCount > 26 then
	-- Expand to three matrix columns (add 40 pixels)
--	windowX = windowX - 5
--	windowW = ui.screenWidth - windowX - 5
end

if itemCount > 39 then
	-- Shrink font to fit more
	priceFont[2] = kSixteen
	countFont[2] = kFourteen
end

if itemCount > 42 then
	-- Expand window downwards as needed to fit more matrix rows
	windowH = windowH + bsgFloor((itemCount - 42) / 3 + .9) * (bsgFontHeight(priceFont) + 2)
end

-- Make small price font even smaller...
local smallPriceSize = bsgFloor(priceFont[2] * .8)
local smallPriceString = "<font size='"..smallPriceSize.."'>"

local charX = windowX + 2
local charY = windowY - 50

local fieldX = charX + ui.charW
local fieldY = windowY
local fieldW = windowX + windowW - fieldX - ui.devButtonW
local fieldH = windowH

-------------------------------------------------------------------------------
-- Price tracking

local function TrackPrices()
	for _,item in ipairs(gShopItems) do
		if (not item.lowseen) or item.price < item.lowseen then item.lowseen = item.price
		elseif (not item.highseen) or item.price > item.highseen then item.highseen = item.price
		end
		
		if not item.highseen then item.highseen = item.lowseen
		elseif not item.lowseen then item.lowseen = item.highseen
		end
	end
end

-------------------------------------------------------------------------------
-- Haggling

local function EnableHaggle()
	local enable = false
	if not (char:IsAngry() or self.owned or gSim.rank==0) then
		-- Verify there are products in inventory
		for item in LItem:AllProducts() do
			if item.inventory > 0 then
				enable = true
				break
			end
		end
	end
	EnableWindow("haggle", enable)
end

local function Haggle()
	local action = DisplayDialog { "ui/haggle.lua", shop=self, char=char }
	EnableHaggle()
	local newColor = nil
	if action == "up" then newColor = BetterPriceColor
	elseif action == "down" then newColor = WorsePriceColor
	end
	if newColor then
		TrackPrices()
		for _,item in ipairs(gShopItems) do
--			if item.inventory == 0 then
--				item.lastprice = item.price
--				item.lastport = gSim.port.name
--			end

			local price = bsgDollars(item.price)
			if item.price > 9999 then price = smallPriceString..price end
			SetLabel(item.name.."_price", newColor..price)
		end
	end
end

-------------------------------------------------------------------------------

local function CloseDialog()
	gShopItems = nil
	CloseWindow()
	if transactionMade then gSim:TickDay() end
end

-------------------------------------------------------------------------------

local function EnterNumber(name, x, y)
	local item = LItem:ByName(name)
	local count = DisplayDialog { "ui/buysell.lua", item=item }
	count = tonumber(count) or 0
	if count > item.inventory then count = item.inventory end
	if count > 0 then
		PlaySound("sell")
	
		item.inventory = item.inventory - count
		item.lastprice = item.price
		item.lastport = gSim.port.name
		item.usetime = gSim.weeks
		item.sold = item.sold + count
		local cost = count * item.price
		local portName = GetString(gSim.port.name)
		gSim:QueueMessage(GetString("sell_notice", tostring(count), GetString(item.name), bsgDollars(item.price), portName))

		transactionMade = true
		-- Once per week a transaction will raise this character's mood
		if char.moodWeek ~= gSim.weeks then
			char:SetMood(char.mood + LCharacter.kPurchaseDelta)
			char.moodWeek = gSim.weeks
		end
		gSim:AdjustMoney(cost)
		gSim:InventoryChanged()
		gSim:FlushMessages()
	end

	EnableHaggle()
end

local function SellAll()
--	local yn = DisplayDialog { "ui/yndialog.lua", body="sellall_confirm" }
--	if yn == "yes" then
		local portName = GetString(gSim.port.name)
		local totalSale = 0
		for _,item in ipairs(gShopItems) do
			local count = item.inventory or 0
			if count > 0 then
				item.inventory = item.inventory - count
				item.lastprice = item.price
				item.lastport = gSim.port.name
				item.usetime = gSim.weeks
				item.sold = item.sold + count
				totalSale = totalSale + count * item.price
				gSim:QueueMessage(GetString("sell_notice", tostring(count), GetString(item.name), bsgDollars(item.price), portName))
			end
		end
		
		if totalSale > 0 then
			PlaySound("sell")
			
			transactionMade = true
			-- Once per week a transaction will raise this character's mood
			if char.moodWeek ~= gSim.weeks then
				char:SetMood(char.mood + LCharacter.kPurchaseDelta)
				char.moodWeek = gSim.weeks
			end
			gSim:AdjustMoney(totalSale)
			gSim:InventoryChanged()
			gSim:FlushMessages()
		end

		CloseDialog()
--	end
end

-------------------------------------------------------------------------------
-- Lay out items as a matrix of columns

local function LayoutMatrix()
	local rowHeight = bsgFontHeight(priceFont) + 2
	local rowCount = itemCount
	local colCount = 1

	local scale = (rowHeight - 2) / ui.itemHeight
	local dxItem = bsgFloor(ui.itemWidth * scale) + 1
	local dxPrice = 3 * rowHeight
	local rowWidth = dxItem+dxPrice+40
	
	if itemCount > 26 then colCount = 3
	elseif itemCount > 8 then colCount = 2
	end

	rowCount = bsgFloor(rowCount / colCount + .9)
	matrixH = rowCount * rowHeight
	local dxColumn = fieldW / colCount

	local matrixW = rowWidth * colCount
	
	fieldY = fieldY + 10		-- Top row must be completely visible
	
	local xLeft = (fieldX - windowX) + (fieldW - matrixW) / (colCount + 1)
	local yTop = (fieldY - windowY) + (fieldH - matrixH) / 2
	
	dxPrice = dxColumn - dxItem
	
	local x = xLeft
	local y = yTop
	local count = 0
	for i,item in ipairs(gShopItems) do
		local gx = windowX + x + dxItem + dxPrice + 20 - 92
		local gy = windowY + y - 14
		local iname = item.name
		local price = bsgDollars(item.price)
		if item.price > 999999 then price = "#"..smallPriceString..price
		else price = "#"..price
		end
		table.insert(items, Group {
			item:Rollover(x,y, scale, "click_sell", function() EnterNumber(iname, gx, gy) end),
			Text { x=x+dxItem,y=y,w=dxPrice,h=rowHeight, name=iname.."_price", font=priceFont, label=price, flags=kVAlignCenter+kHAlignLeft },
--			Rectangle { x=x+dxItem+dxPrice,y=y,w=40,h=rowHeight-2 },
--			Button { x=x+dxItem+dxPrice,y=y,w=40,h=rowHeight-2, graphics = {},
--				command = function() EnterNumber(iname,gx,gy) end,
--				TextEdit { x=0,y=0,w=kMax,h=kMax, name=iname, label=GetString("lbl_sell"), font=countFont, length=5, ignore=kNumbersOnly },
--			},
		})
		
		y = y + rowHeight
		count = count + 1
		if count == rowCount then
			x = x + dxColumn
			y = yTop
			count = 0
		end
	end
end

-------------------------------------------------------------------------------
-- lay out items as a bunch of crates

local function LayoutCrates()
	local itemWidth = ui.crateWidth + 2
	local itemHeight = ui.crateHeight + 37 + bsgFontHeight(countFont) + 5 - 10
	local itemsPerRow = bsgFloor(fieldW / itemWidth)
	
	-- Force two rows with more than three crates
	if itemCount > 3 and itemCount <= itemsPerRow then
		itemsPerRow = itemsPerRow / 2
	end

	-- Determine how many rows and how many items to place on each row
	local colCount = {}
	local rowCount = 1
	if itemCount > itemsPerRow then
		local total = itemCount
		rowCount = bsgFloor((itemCount + itemsPerRow - 1) / itemsPerRow)
		for i=1,rowCount do
			local count = bsgFloor((itemCount + rowCount - 1) / rowCount)
			if count > total then count = total end
			colCount[i] = count
			total = total - count
		end
	else
		colCount[1] = itemCount
	end

	local y = fieldY - windowY + (fieldH - itemHeight * rowCount) / 2

	local n = 0
	local row = 1
	local x = fieldX - windowX + (fieldW - colCount[row] * itemWidth) / 2

	local offsetX = (ui.crateWidth - ui.itemWidth) / 2
	local offsetY = (ui.crateHeight - ui.itemHeight) / 2 + 4
	
	priceFont[2] = kSixteen

	for _,item in ipairs(gShopItems) do
		local gx = windowX + x - 67
		local gy = windowY + y + 67
		local iname = item.name
		local price = bsgDollars(item.price)
		table.insert(items,
			Window
			{
				x=x,y=y,w=itemWidth,h=itemHeight, fit=true,
				Bitmap { x=kCenter,y=0, image="image/pricetag",
					Text { x=0,y=3,w=kMax,h=20, font=priceFont, name=item.name.."_price", label="#"..price },
				},
				Bitmap { x=0,y=37, image="image/icon_crate",
					item:Rollover(offsetX,offsetY, 1, "click_sell", function() EnterNumber(iname,gx,gy) end),
				},
--				Rectangle { x=5,y=84,w=40,h=bsgFontHeight(countFont)+2 },
--				Button { x=5,y=84,w=40,h=bsgFontHeight(countFont)+2,
--					graphics = {},
--					command = function() EnterNumber(iname,gx,gy) end,
--					TextEdit { x=0,y=0,w=kMax,h=kMax, name = item.name, label=GetString("lbl_sell"), font=countFont, length=5, ignore=kNumbersOnly },
--				},
			})

		x = x + itemWidth
		n = n + 1
		if n == colCount[row] and row < rowCount then
			n = 0
			row = row + 1
			y = y + itemHeight
			x = fieldX - windowX + (fieldW - colCount[row] * itemWidth) / 2
		end
	end
end

-------------------------------------------------------------------------------
-- lay out items on large platters

local function LayoutLargePlatters()
	local itemWidth = ui.platterWidth + 2
	local itemHeight = ui.platterHeight + 8
--	local itemsPerRow = bsgFloor(fieldW / itemWidth)
	local itemsPerRow = 2	--bsgFloor(fieldW / itemWidth)
	
	-- Force two rows with more than two platters
	if itemCount > 2 and itemCount <= itemsPerRow then
		itemsPerRow = itemsPerRow / 2
	end

	-- Determine how many rows and how many items to place on each row
	local colCount = {}
	local rowCount = 1
	if itemCount > itemsPerRow then
		local total = itemCount
		rowCount = bsgFloor((itemCount + itemsPerRow - 1) / itemsPerRow)
		for i=1,rowCount do
			local count = bsgFloor((itemCount + rowCount - 1) / rowCount)
			if count > total then count = total end
			colCount[i] = count
			total = total - count
		end
	else
		colCount[1] = itemCount
	end

	local y = fieldY - windowY + (fieldH - itemHeight * rowCount) / 2

	local n = 0
	local row = 1
	local x = fieldX - windowX + (fieldW - colCount[row] * itemWidth) / 2

	local offsetX = (ui.platterWidth - 2 * ui.itemWidth) / 2
	local offsetY = 8

	for _,item in ipairs(gShopItems) do
		local gx = windowX + x - 8
		local gy = windowY + y + 114
		local iname = item.name
		local price = bsgDollars(item.price)
		if item.price > 9999 then price = smallPriceString..price end
		table.insert(items,
			Window
			{
				x=x,y=y,w=itemWidth,h=itemHeight, fit=true,
				Bitmap { x=0,y=8, image="image/platter",
					Text { x=53,y=86,w=60,h=29, font=priceFont, name=item.name.."_price", label="#"..price },
				},
				item:BigRollover(offsetX,offsetY, 1, "click_sell", function() EnterNumber(iname,gx,gy) end),
				Button { x=61,y=131,w=47,h=bsgFontHeight(countFont),
					graphics = {},
					command = function() EnterNumber(iname,gx,gy) end,
					TextEdit { x=0,y=0,w=kMax,h=kMax, name = item.name, label=GetString("lbl_sell"), font=countFont, length=5, ignore=kNumbersOnly },
				},
			})

		x = x + itemWidth
		n = n + 1
		if n == colCount[row] and row < rowCount then
			n = 0
			row = row + 1
			y = y + itemHeight
			x = fieldX - windowX + (fieldW - colCount[row] * itemWidth) / 2
			if itemCount > 3 then x = x - 30 end
		end
	end
end

-------------------------------------------------------------------------------
-- lay out items on small platters

local function LayoutSmallPlatters()
	local itemWidth = (ui.platterWidth/2) + 2
	local itemHeight = (ui.platterHeight/2) + 12
	local itemsPerRow = bsgFloor(fieldW / itemWidth)
	
	-- Force two rows with more than four platters
	if itemCount > 2 and itemCount <= itemsPerRow then
		itemsPerRow = itemsPerRow / 2
	end

	-- Determine how many rows and how many items to place on each row
	local colCount = {}
	local rowCount = 1
	if itemCount > itemsPerRow then
		local total = itemCount
		rowCount = bsgFloor((itemCount + itemsPerRow - 1) / itemsPerRow)
		for i=1,rowCount do
			local count = bsgFloor((itemCount + rowCount - 1) / rowCount)
			if count > total then count = total end
			colCount[i] = count
			total = total - count
		end
	else
		colCount[1] = itemCount
	end

	local y = fieldY - windowY + (fieldH - itemHeight * rowCount) / 2

	local n = 0
	local row = 1
	local x = fieldX - windowX + (fieldW - colCount[row] * itemWidth) / 2 + 20

	local offsetX = (ui.platterWidth/2 - ui.itemWidth) / 2
	local offsetY = 5

	for _,item in ipairs(gShopItems) do
		local gx = windowX + x - 48
		local gy = windowY + y + 50
		local iname = item.name
		local price = bsgDollars(item.price)
		if item.price > 9999 then price = smallPriceString..price end
		table.insert(items,
			Window
			{
				x=x,y=y,w=itemWidth,h=itemHeight, fit=true,
				Bitmap { x=0,y=8, image="image/platter_half",
					Text { x=17,y=33,w=50,h=25, font=priceFont, name=item.name.."_price", label="#"..price },
				},
				item:Rollover(offsetX,offsetY, 1, "click_sell", function() EnterNumber(iname,gx,gy) end),
				Button { x=19,y=67,w=48,h=bsgFontHeight(countFont),
					graphics = {},
					command = function() EnterNumber(iname,gx,gy) end,
					TextEdit { x=0,y=0,w=kMax,h=kMax, name = item.name, label=GetString("lbl_sell"), font=countFont, length=5, ignore=kNumbersOnly },
				},
			})

		x = x + itemWidth
		n = n + 1
		if n == colCount[row] and row < rowCount then
			n = 0
			row = row + 1
			y = y + itemHeight
			x = fieldX - windowX + (fieldW - colCount[row] * itemWidth) / 2 + 20
			if row == 2 then x = x - 20 end
		end
	end
end

-------------------------------------------------------------------------------
--

if itemCount <= 2 then LayoutLargePlatters()
elseif itemCount <= 8 then LayoutSmallPlatters()
elseif itemCount >= 15 then LayoutMatrix()
else LayoutCrates()
end

TrackPrices()

local buttonX = windowW - ui.devButtonW
local buttonY = charY + ui.charH - windowY - 80		--100
MakeDialog
{
	name=self.name,
	DialogLedger(),
	Frame
	{
		x=windowX,y=windowY,w=windowW,h=windowH,fit=true, color=ShopColor,

		Group(items),

		TextButton { x=buttonX,y=buttonY, name="haggle", label="haggle", command=Haggle, },
		TextButton { x=buttonX,y=buttonY+40, name="sell_all", label="sellall", command=SellAll, },
		TextButton { x=buttonX,y=buttonY+80, name="help", label="help", graphics=SmallButtonGraphics, command=function() HelpDialog(5) end },
		TextButton { x=buttonX+62,y=buttonY+80, name="ok", label="close", command=CloseDialog, graphics=MediumButtonGraphics },
	},
	BuildingBanner(self),
	char:Visual(charX, charY, charNameFontDark),
}

EnableHaggle()
